/*******************************************************************************
 * Copyright (c) 2018 Agence spatiale canadienne / Canadian Space Agency 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Mathieu Larose (Savoir-faire Linux) - Initial API and implementation
 *
 * SPDX-License-Identifier: EPL-1.0
 *     
 *******************************************************************************/
package org.eclipse.apogy.core.programs.javascript;

import org.eclipse.apogy.common.emf.ApogyCommonTransactionFacade;
import org.eclipse.apogy.core.invocator.AbstractOperationCall;
import org.eclipse.apogy.core.invocator.ApogyCoreInvocatorFacade;
import org.eclipse.apogy.core.invocator.ApogyCoreInvocatorFactory;
import org.eclipse.apogy.core.invocator.ApogyCoreInvocatorPackage;
import org.eclipse.apogy.core.invocator.ArgumentsList;
import org.eclipse.apogy.core.invocator.NumericEDataTypeArgument;
import org.eclipse.apogy.core.invocator.OperationCallResult;
import org.eclipse.apogy.core.invocator.VariableFeatureReference;
import org.eclipse.emf.ecore.EOperation;
import org.mozilla.javascript.BaseFunction;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.Scriptable;

/**
 * Calls an operation through the {@link ApogyCoreInvocatorFacade}
 *
 */
public class OperationCallInvoker extends BaseFunction {
	private static final long serialVersionUID = 1L;
	private final EOperation operation;
	private final JavaScriptProgram program;
	private final VariableFeatureReference variableFeatureReference;

	/**
	 *
	 * @param program                  The JavaScriptProgram being executed
	 * @param variableFeatureReference The variable, type member and feature
	 * @param operation                The operation to execute
	 */
	public OperationCallInvoker(JavaScriptProgram program, VariableFeatureReference variableFeatureReference,
			EOperation operation) {
		this.program = program;
		this.operation = operation;
		this.variableFeatureReference = variableFeatureReference;
	}

	/**
	 * Converts Rhino arguments to Apogy arguments.
	 *
	 * @param args Rhino arguments
	 * @return Apogy arguments
	 */
	private static ArgumentsList createArgumentsList(Object[] args) {
		ArgumentsList argumentsList = ApogyCoreInvocatorFactory.eINSTANCE.createArgumentsList();
		for (Object arg : args) {
			NumericEDataTypeArgument argument = ApogyCoreInvocatorFactory.eINSTANCE.createNumericEDataTypeArgument();
			argument.setValue(Context.toString(arg));
			argumentsList.getArguments().add(argument);
		}

		return argumentsList;
	}

	/**
	 * Creates an {@link AbstractOperationCall} based on the private attributes and
	 * arguments.
	 *
	 * @param argumentsList Apogy arguments list for the operation call.
	 * @return an {@link AbstractOperationCall}
	 */
	private AbstractOperationCall createOperationCall(ArgumentsList argumentsList) {
		AbstractOperationCall opCall = VariableFeatureReferenceUtil.toOperationCall(this.variableFeatureReference);
		opCall.setEOperation(this.operation);
		opCall.setArgumentsList(argumentsList);

		// Add the opCall using a transaction.
		ApogyCommonTransactionFacade.INSTANCE.basicAdd(this.program,
				ApogyCoreInvocatorPackage.Literals.OPERATION_CALL_CONTAINER__OPERATION_CALLS, opCall, true);

		return opCall;
	}

	/**
	 * Invokes an operation call.
	 *
	 * @param cx      Rhino context
	 * @param scope   Scope of the method called
	 * @param thisObj Object of the method called
	 * @param args    Arguments of the method called
	 * @return the value of the {@link OperationCallResult}
	 */
	@Override
	public Object call(Context cx, Scriptable scope, Scriptable thisObj, Object[] args) {
		ArgumentsList argumentsList = createArgumentsList(args);
		AbstractOperationCall opCall = createOperationCall(argumentsList);

		OperationCallResult opCallResult = ApogyCoreInvocatorFacade.INSTANCE.exec(opCall, true);
		Object value = ApogyCoreInvocatorFacade.INSTANCE.getValue(opCallResult);

		return Context.javaToJS(value, scope);
	}
}